
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2004
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_AES.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"

#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_AES_Acceptance.h"
#include "CRYS_ATP_TST_AES_Acceptance_test_vectors.h"

#include "DX_VOS_Mem.h"

#if TST_USE_AES_DATA
/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/
static CRYS_AES_EncryptMode_t TST_EncryptMode[] = { CRYS_AES_Encrypt , CRYS_AES_Decrypt };
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_NON_integrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES non integrated API's                         
 *  This function Encrypts and Decrypts known vectors using AES
 *  AES modes                   - CBC/ECB/CTR   
 *  Vector sizes used (in bytes) - 16/32/512                                 
 *  Key Size used (in bits)     - 128/192/256                               
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 * 1. Call the CRYS_AES_Init function to start the encryption/decryption process
 * 2. Call the CRYS_AES_Block - This function is called N times 
 *                              (N = number of blocks in data each block is 16byte) 
 * 3. Call the CRYS_AES_Free to end the encryption/decryption process 
 * 4. Compare the outputted result to the expected one                                                   
 *
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_AES_Acceptance_NON_integrated ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     TST_index;
    int                     Block_index ,NumOfBlocks;
    CRYSError_t             TST_Status=0;
    int                     TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    CRYS_AESUserContext_t   TST_ContextID_ptr;
    char                    TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_NON_integrated"; 

    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_NON_integrated Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************* \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_AES_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
            for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
                                                                                                          TST_OperationMode = 0 == Encrypt
                                                                                                          TST_OperationMode = 1 == Decrypt*/
            {          
                if (
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_ECB_mode
                        ||
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_CBC_mode
                        ||
                        (
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_CTR_mode &&
                        TST_ATP_EncryptDecryptFlag == TST_ENCRYPT /*CTR mode uses encrypt only!*/
                        )
                   )
                {                                               
                    if (TST_OutputLocation == 0) /*Select output buffer*/
                    {   
                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 1A.%d] \n",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_index ));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   output buffer != input buffer                                       \n"));         
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
                    }
                    else
                    {
                        /* output buffer == input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 1B.%d] \n",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_index));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   output buffer == input buffer (output buffer)                       \n"));                             
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n "));
                    }
                    
                    TST_Status = TESTPASS;
                            
                    /* calling the AES init function */ 
                    /*-------------------------------*/ 
                    TST_Status = CRYS_AES_Init(&TST_ContextID_ptr,
                                                  TST_AES_Acceptance_TestData[TST_index].TST_IvCounter, 
                                                  TST_AES_Acceptance_TestData[TST_index].TST_Key,
                                                  TST_AES_Acceptance_TestData[TST_index].TST_KeySize,
                                                  TST_EncryptMode[TST_ATP_EncryptDecryptFlag],  
                                                  TST_AES_Acceptance_TestData[TST_index].TST_OperationMode);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Init",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

                    
                    /* calling the AES Block function */
                    /*--------------------------------*/
                    NumOfBlocks = TST_AES_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
                    
                    /* Debug massages */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %d \n",NumOfBlocks));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %d \n" ,TST_AES_Acceptance_TestData[TST_index].TST_dataSize));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %d \n",TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS));
                    
                    
                    if (TST_OutputLocation == 0)/* input buffer != output buffer */
                    {
                        for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*AES block loop*/ 
                        {   
                            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                            { 
                                TST_Status = CRYS_AES_Block( &TST_ContextID_ptr,     
                                                           TST_AES_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                                                           TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                                           TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block - Encrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);                             
                            }
                            else
                            {                           
                                TST_Status = CRYS_AES_Block( &TST_ContextID_ptr,     
                                                           TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),         
                                                           TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                                           TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block - Decrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
                            }   
 
                                          
                        }/*End AES block loop*/ 
                    }
                    else /* input buffer = output buffer */ 
                    {
                       for (Block_index =0 ; Block_index < NumOfBlocks  ; Block_index++)
                        {
                            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                            {
                                /*Copying the input buffer onto output buffer not to overwrite test data*/
                                DX_VOS_FastMemCpy(TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                                  TST_AES_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                                  TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                            }
                            else
                            {
                                /*Copying the input buffer onto output buffer not to overwrite test data*/
                                DX_VOS_FastMemCpy(TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                               TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index), 
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                            }
                        }           

                        for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                        {   
                            TST_Status = CRYS_AES_Block(&TST_ContextID_ptr,           
                                                       TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                                                       TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                                       TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        
                                
                            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
              {
                               TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block Encrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
              }
                            else
              {
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block Decrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
              }

                        
                        }
                    }    

                    
                    /* calling the Free function */ 
                    /*---------------------------*/     
                    TST_Status = CRYS_AES_Free( &TST_ContextID_ptr );
                    
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Free",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
                    
                     
                        
                                
                    /* Compare the result to expected result */ 
                    /*---------------------------------------*/ 
                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)  
                    {
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_TestData[TST_index].TST_output_data ,
                                                          TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data, 
                                                          TST_AES_Acceptance_TestData[TST_index].TST_dataSize);
                    }
                    else
                    {
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_TestData[TST_index].TST_output_data ,
                                                        TST_AES_Acceptance_TestData[TST_index].TST_input_data,
                                                        TST_AES_Acceptance_TestData[TST_index].TST_dataSize);                
                    }
                    
                    /*Print the outputted and then the expected result after mem compare */
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "\nData outputted by engine [output buffer]-- \n",
                                                TST_AES_Acceptance_TestData[TST_index].TST_output_data, 
                                                TST_AES_Acceptance_TestData[TST_index].TST_dataSize); 

                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                    {
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                    "\nData Expected by engine [expected buffer - encrypt]--\n",
                                                    TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data,
                                                    TST_AES_Acceptance_TestData[TST_index].TST_dataSize);                 
                    }
                    else
                    {
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                    "\nData Expected by engine [input buffer - for decrypt]--\n",
                                                    TST_AES_Acceptance_TestData[TST_index].TST_input_data,
                                                    TST_AES_Acceptance_TestData[TST_index].TST_dataSize);                     
                    }           

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);                   
                    
                     
                    
                } /* End of first if statement*/ 
            }/* End of TST_ATP_EncryptDecryptFlag*/ 
        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;

  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/* End of CRYS_ATP_TST_AES_Acceptance_NON_integrated*/ 


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_Integrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES integrated API                           
 *  This function Encrypts and Decrypts known vectors using AES
 *  AES modes                   - CBC/ECB/CTR   
 *  Vector sizes used (in bytes) - 16/32/512                                 
 *  Key Size used (in bits)     - 128/192/256                               
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 * 1. Call the CRYS_AES function to use the AES encryption/decryption process
 * 2. Compare the outputted result to the expected one                                                   
 *
 ******************************************************************************/    
CRYSError_t CRYS_ATP_TST_AES_Acceptance_Integrated ( void ) 
{

    /********* TST Variables Definitions ************/
    int              TST_index; 
    int              Block_index ,NumOfBlocks;
    CRYSError_t      TST_Status=0;
    int              TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    char             TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_Integrated";
    /********* TST Variables initialization *********/
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n")); 
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ** CRYS_ATP_TST_AES_Acceptance_Integrated Tests     ********************* \n"));     
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n \n"));

    
    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_AES_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
            for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++) /*Select mode or operation  
                                                                                                      TST_OperationMode = 0 == Encrypt
                                                                                                      TST_OperationMode = 1 == Decrypt*/
            {
                if (
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_ECB_mode
                        ||
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_CBC_mode
                        ||
                        (
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_CTR_mode &&
                        TST_ATP_EncryptDecryptFlag == TST_ENCRYPT /*CTR mode uses encrypt only!*/
                        )
                   )
                {               
            
            
                    if (TST_OutputLocation == 0)/*Select output buffer*/ 
                    {   
                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************** \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    %s [test 2A.%d]  **\n",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_index));   
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    output buffer != input buffer                                       \n"));                    
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************** \n"));
                    }
                    else
                    {
                        /* output buffer == input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ************************************************************************ \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    %s [test 2B.%d]   **\n",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_index));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    output buffer == input buffer (output buffer)                         \n"));                      
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ************************************************************************ \n"));
                    }
                    
                    TST_Status = TESTPASS;
                    NumOfBlocks = TST_AES_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
                            
                    /* calling the AES integrated function */
                    /*-------------------------------------*/ 
                    if (TST_OutputLocation == 0)/* input buffer != output buffer */
                    {
                        if (TST_ATP_EncryptDecryptFlag ==0) 
                        {
                            TST_ATP_StartPerformanceTimer(&TST_Timer);

                            TST_Status = CRYS_AES(TST_AES_Acceptance_TestData[TST_index].TST_IvCounter,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_Key,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_KeySize,
                                                     TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                                     TST_AES_Acceptance_TestData[TST_index].TST_OperationMode,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_input_data, 
                                                     TST_AES_Acceptance_TestData[TST_index].TST_dataSize, 
                                                     TST_AES_Acceptance_TestData[TST_index].TST_output_data); 

                            TST_ATP_GetPerformanceTimer(&TST_Timer);
                            TST_ATP_CalcPerformance(TST_AES_Acceptance_TestData[TST_index].TST_dataSize);
                        }
                        else
                        {
                            TST_ATP_StartPerformanceTimer(&TST_Timer);

                            TST_Status = CRYS_AES(TST_AES_Acceptance_TestData[TST_index].TST_IvCounter,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_Key,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_KeySize,
                                                     TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                                     TST_AES_Acceptance_TestData[TST_index].TST_OperationMode,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_dataSize,
                                                     TST_AES_Acceptance_TestData[TST_index].TST_output_data);                   

                            TST_ATP_GetPerformanceTimer(&TST_Timer); 
                            TST_ATP_CalcPerformance(TST_AES_Acceptance_TestData[TST_index].TST_dataSize);
                        }
                    }
                    else /* input buffer = output buffer */ 
                    {
                        for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                        {
                            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                            {
                            /*Copying the input buffer onto output buffer not to overwrite test data*/
                                DX_VOS_FastMemCpy(TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                               TST_AES_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                            }
                            else
                            {
                            /*Copying the input buffer onto output buffer not to overwrite test data*/
                                DX_VOS_FastMemCpy(TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                               TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index), 
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                            }
                        }
                        TST_ATP_StartPerformanceTimer(&TST_Timer);

                        TST_Status = CRYS_AES(TST_AES_Acceptance_TestData[TST_index].TST_IvCounter,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_Key,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_KeySize,
                                                 TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                                 TST_AES_Acceptance_TestData[TST_index].TST_OperationMode,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_output_data,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_dataSize,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_output_data);
                        
                        TST_ATP_GetPerformanceTimer(&TST_Timer);
                        TST_ATP_CalcPerformance(TST_AES_Acceptance_TestData[TST_index].TST_dataSize);
                    } 
             
                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
          {
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES Encrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
          }
                    else
          {
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES Decrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
          }


                    
                    /* Debug massages */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"Input Data Size [byte] = %d \n" ,TST_AES_Acceptance_TestData[TST_index].TST_dataSize));
                                
                    /* Compare the result to expected result */ 
                    /*---------------------------------------*/                                                             
                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)  
                    {
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_TestData[TST_index].TST_output_data ,
                                                        TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data ,
                                                        TST_AES_Acceptance_TestData[TST_index].TST_dataSize);
                    }
                    else
                    {
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_TestData[TST_index].TST_output_data ,
                                                        TST_AES_Acceptance_TestData[TST_index].TST_input_data,
                                                        TST_AES_Acceptance_TestData[TST_index].TST_dataSize);                
                    }

                    /*Print the outputted and then the expected result after mem compare */
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "\nData outputted by engine [output buffer]-- \n",
                                                TST_AES_Acceptance_TestData[TST_index].TST_output_data, 
                                                TST_AES_Acceptance_TestData[TST_index].TST_dataSize); 

                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                    {
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                    "\nData Expected by engine [expected buffer - encrypt]--\n",
                                                    TST_AES_Acceptance_TestData[TST_index].TST_Expected_output_data,
                                                    TST_AES_Acceptance_TestData[TST_index].TST_dataSize);                 
                    }
                    else
                    {
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                    "\nData Expected by engine [input buffer - for decrypt]--\n",
                                                    TST_AES_Acceptance_TestData[TST_index].TST_input_data,
                                                    TST_AES_Acceptance_TestData[TST_index].TST_dataSize);                     
                    }               
                    
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);  

                }/* End of first if statment*/
            }/* End of TST_ATP_EncryptDecryptFlag*/ 
        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;
  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/* End of CRYS_ATP_TST_AES_Acceptance_Integrated */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_MAC_NON_integrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES non integrated API's                         
 *  This function Encrypts and Decrypts known vectors using AES
 *  AES modes                   - MAC   
 *  Vector sizes used (in bytes) - 16/32/512/1536                     
 *  Key Size used (in bits)     - 128/192/256                               
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 * 1. Call the CRYS_AES_Init function to start the encryption/decryption process
 * 2. Call the CRYS_AES_Block - This function is called N times 
 *                              (N = number of blocks in data each block is 16byte) 
 * 3. Call the CRYS_AES_Free to end the encryption/decryption process 
 * 4. Compare the outputted result to the expected one                                                   
 *
 ******************************************************************************/
CRYSError_t CRYS_ATP_TST_AES_Acceptance_MAC_NON_integrated ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     TST_index;
    int                     Block_index ,NumOfBlocks;
    CRYSError_t             TST_Status=0;
    int                     Index;
    int                     TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    char                    TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_MAC_NON_integrated"; 
    CRYS_AESUserContext_t   TST_ContextID_ptr;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*****************************************************************************\n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_NON_integrated CBC MAC Tests *********************\n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*****************************************************************************\n\n")); 

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_ATP_EncryptDecryptFlag = 0; 
    
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
    {
        for (TST_index =0 ; TST_index < TST_NumberOfMACAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_OutputLocation == 0) 
            {   
                /* output buffer != input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3A.%d]\n",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_index )); 
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer != input buffer                                     \n"));        
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
            }
            else
            {
                /* output buffer == input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3B.%d]\n",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_index));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer == input buffer (output buffer)                      \n"));                           
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
            }
            
            TST_Status = TESTPASS;
                    
            /* calling the AES init function */ 
            /*-------------------------------*/ 
            TST_Status = CRYS_AES_Init(&TST_ContextID_ptr,
                                      TST_AES_MAC_Acceptance_TestData[TST_index].TST_IvCounter, 
                                      TST_AES_MAC_Acceptance_TestData[TST_index].TST_Key,
                                      TST_AES_MAC_Acceptance_TestData[TST_index].TST_KeySize,
                                      TST_EncryptMode[TST_ATP_EncryptDecryptFlag],  
                                      TST_AES_MAC_Acceptance_TestData[TST_index].TST_OperationMode);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Init",
                                       TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,
                                       TST_MasterTestName);
            
            /* calling the AES Block function */
            /*--------------------------------*/
            NumOfBlocks = TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
            
            /* Debug massages */
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"number of blocks = %d \n",NumOfBlocks)); 
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"Input Data Size [byte] = %d \n" ,TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"SingleBlockSize [byte] = %d \n",TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS)); 
            
            
            if (TST_OutputLocation == 0) 
            {
                for (Block_index = 0 ; Block_index < NumOfBlocks    ; Block_index++)
                {   
                    TST_Status = CRYS_AES_Block( &TST_ContextID_ptr,     
                                               TST_AES_MAC_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                               TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
                                   
                }
            }
            else /* input buffer = output buffer */ 
            {
                for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                {
                    /*Copying the input buffer onto output buffer not to overwrite test data*/
                    DX_VOS_FastMemCpy(TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                   TST_AES_MAC_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                   TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                }           

                for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                {   
                    TST_Status = CRYS_AES_Block(&TST_ContextID_ptr,           
                                               TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                               TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        
                                          
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

                }
            
            }    
  
            /* calling the Free function */ 
            /*---------------------------*/     
      
         TST_Status = CRYS_AES_Free( &TST_ContextID_ptr );
       
          
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Free",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

                
                        
            /* Compare the result to expected result */ 
            /*---------------------------------------*/ 

            for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
            {
                if (TST_AES_MAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC[Index] !=
                    TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data[((NumOfBlocks-1)*TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS)+Index])
                        TST_Status = TESTFAIL;
            }
            
            /*Print the outputted and then the expected result after mem compare */
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data, 
                                        TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize); 
                                        
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\n Data Expected by engine [expected buffer]  -- \n",
                                        TST_AES_MAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC,
                                        TST_CBC_MAC_SIZE);                                      

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_MemCompare",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;

  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
} /***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_XCBC_CMAC_NON_integrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES non integrated API's                         
 *  This function Encrypts and Decrypts known vectors using AES new modes
 *  XCBC and CMAC
 *  AES modes                   - MAC   
 *  Vector sizes used (in bytes) - 16/32/512/1536                     
 *  Key Size used (in bits)     - 128/ 192 CMAC only / 256 CMAC only                              
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 * 1. Call the CRYS_AES_Init_XCBCandCMAC function to start the HASH process
 * 2. Call the CRYS_AES_Block - This function is called N times 
 *                              (N = number of blocks in data each block is 16byte) 
 * 3. Call the CRYS_AES_XCBCandCMAC_Finish to end the encryption/decryption process 
 * 4. Compare the outputted result to the expected one                                                   
 *
 ******************************************************************************/
CRYSError_t CRYS_ATP_TST_AES_Acceptance_XCBC_CMAC_NON_integrated ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     TST_index;
    int                     Block_index ,NumOfBlocks;
    CRYSError_t             TST_Status=0;
    int                     Index;
    int                     TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    char                    TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_XCBC_CMAC_NON_integrated"; 
    CRYS_AESUserContext_t   TST_ContextID_ptr;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"***************************************************************************************\n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_NON_integrated XCBC/CMAC MAC Tests *********************\n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"***************************************************************************************\n\n")); 

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_ATP_EncryptDecryptFlag = 0; 
    
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
    {
        for (TST_index =0 ; TST_index < TST_NumberOfXCBCMACAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {

            if (TST_OutputLocation == 0) 
            {   
                /* output buffer != input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3A.%d]\n",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_index )); 
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer != input buffer                                     \n"));        
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
            }
            else
            {
                /* output buffer == input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3B.%d]\n",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_index));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer == input buffer (output buffer)                      \n"));                           
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
            }
            
            TST_Status = TESTPASS;
                    
            /* calling the AES init function */ 
            /*-------------------------------*/ 
      if (TST_USE_SECRETKEY_TESTS) 
            {
          TST_Status = CRYS_AES_Init_XCBCandCMAC_SecretKey(&TST_ContextID_ptr,
                                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_OperationMode);
            }
            else
            {
          TST_Status = CRYS_AES_Init_XCBCandCMAC(&TST_ContextID_ptr,
                                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Key,
                                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_KeySize,
                                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_OperationMode);
            }


            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Init",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 
            
            /* calling the AES Block function */
            /*--------------------------------*/

      NumOfBlocks = TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
            
            /* Debug massages */
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"number of blocks = %d \n",NumOfBlocks)); 
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"Input Data Size [byte] = %d \n" ,TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"SingleBlockSize [byte] = %d \n",TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS)); 
            
            
            if (TST_OutputLocation == 0) 
            {
                for (Block_index = 0 ; Block_index < NumOfBlocks    ; Block_index++)
                {   
                    TST_Status = CRYS_AES_Block( &TST_ContextID_ptr,     
                                               TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                               TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
                                  
                }
            }
            else /* input buffer = output buffer */ 
            {
               
                DX_VOS_FastMemCpy(TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data, 
                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_input_data,
                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize);

                for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                {   
                    TST_Status = CRYS_AES_Block(&TST_ContextID_ptr,           
                                               TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                                               TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        
                                          
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

                }
            
            }    
  
            /* calling the Finish function */ 
            /*---------------------------*/     
                   
      TST_Status = CRYS_AES_XCBCandCMAC_Finish( &TST_ContextID_ptr,TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data );
          
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Free",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
                
                        
            /* Compare the result to expected result */ 
            /*---------------------------------------*/ 

            for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
            {
                if (TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC[Index] !=
                    TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data[Index])
                        TST_Status = TESTFAIL;
            }
            
            /*Print the outputted and then the expected result after mem compare */
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data, 
                                        TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize); 
                                        
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\n Data Expected by engine [expected buffer]  -- \n",
                                        TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC,
                                        TST_CBC_MAC_SIZE);                                      

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_MemCompare",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
            
        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;

  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
} 
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_XCBC_CMAC_integrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES non integrated API's                         
 *  This function Encrypts and Decrypts known vectors using AES new modes
 *  XCBC and CMAC
 *  AES modes                   - XCBX/CMAC   
 *  Vector sizes used (in bytes) - 16/32/512/1536                     
 *  Key Size used (in bits)     - 128/192 CMAC only/256 CMAC only/Secret key 
 *  
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 * 1. Call the CRYS_AES_XCBCandCMAC or CRYS_AES_XCBCandCMAC_SecretKey function to start the HASH process
 * 2. Compare the outputted result to the expected one                                                   
 *
 ******************************************************************************/
CRYSError_t CRYS_ATP_TST_AES_Acceptance_XCBC_CMAC_integrated ( void ) 
{

    /********* TST Variables Definitions ************/
    int                     TST_index;    
    CRYSError_t             TST_Status=0;
    int                     Index;
    int                     TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    char                    TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_XCBC_CMAC_NON_integrated"; 
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"***************************************************************************************\n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_integrated XCBC/CMAC Tests *********************\n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"***************************************************************************************\n\n")); 

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_ATP_EncryptDecryptFlag = 0; 
    
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
    {
        for (TST_index =0 ; TST_index < TST_NumberOfXCBCMACAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {

            if (TST_OutputLocation == 0) 
            {   
                /* output buffer != input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3A.%d]\n",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_index )); 
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer != input buffer                                     \n"));        
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
            }
            else
            {
                /* output buffer == input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3B.%d]\n",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_index));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer == input buffer (output buffer)                      \n"));                           
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
            }
            
            TST_Status = TESTPASS;
            /* Calling the AES function */
            /****************************/                        
            if (TST_OutputLocation == 0) 
            {
          if (TST_USE_SECRETKEY_TESTS) 
                {
              TST_Status = CRYS_AES_XCBCandCMAC_SecretKey(TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_OperationMode,
                                                                   TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_input_data,
                                                                   TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize,
                                                                   TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XCBCandCMAC_SecretKey",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

                }
                else
                {
              TST_Status = CRYS_AES_XCBCandCMAC(TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Key,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_KeySize,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_OperationMode,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_input_data,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XCBCandCMAC",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
  
                } 
            }
            else /* input buffer = output buffer */ 
            {              
                
                DX_VOS_FastMemCpy(TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data, 
                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_input_data,
                          TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize); 

          if (TST_USE_SECRETKEY_TESTS) 
                {
              TST_Status = CRYS_AES_XCBCandCMAC_SecretKey(TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_OperationMode,
                                                                   TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data,
                                                                   TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize,
                                                                   TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XCBCandCMAC_SecretKey",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

                }
                else
                {
              TST_Status = CRYS_AES_XCBCandCMAC(TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Key,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_KeySize,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_OperationMode,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize,
                                                         TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XCBCandCMAC",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
  
                } 
            }            
                        
            /* Compare the result to expected result */ 
            /*---------------------------------------*/ 

            for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
            {
                if (TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC[Index] !=
                    TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data[Index])
                        TST_Status = TESTFAIL;
            }
            
            /*Print the outputted and then the expected result after mem compare */
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_output_data, 
                                        TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_dataSize); 
                                        
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\n Data Expected by engine [expected buffer]  -- \n",
                                        TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC,
                                        TST_CBC_MAC_SIZE);                                      

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_MemCompare",TST_AES_XCBC_CMAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;


  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
} 
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_MAC_integrated      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES non integrated API's                         
 *  This function Encrypts and Decrypts known vectors using AES
 *  AES modes                   - MAC   
 *  Vector sizes used (in bytes) - 16/32/512/1536                           
 *  Key Size used (in bits)     - 128/192/256
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 * 1. Call the CRYS_AES_Init function to start the encryption/decryption process
 * 2. Compare the outputted result to the expected one                                                   
 *
 ******************************************************************************/
CRYSError_t CRYS_ATP_TST_AES_Acceptance_MAC_integrated ( void ) 
{

    /********* TST Variables Definitions ************/
    int              TST_index,Index;
    int              Block_index ,NumOfBlocks;
    CRYSError_t      TST_Status=0;
    int              TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    char             TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_MAC_integrated";
    
        
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************************************************************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_Integrated CBC MAC Tests ********************* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************************************************************* \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_ATP_EncryptDecryptFlag = 0; 
        
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
    {
        for (TST_index =0 ; TST_index < TST_NumberOfMACAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {

            if (TST_OutputLocation == 0) 
            {   
                /* output buffer != input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 4A.%d]\n",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_index )); 
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer != input buffer                                      \n"));       
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
            }
            else
            {
                /* output buffer == input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 4B.%d]\n",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_index));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer == input buffer (output buffer)                      \n"));                           
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"**********************************************************************\n"));
            }
            
            TST_Status = TESTPASS;
                    
            NumOfBlocks = TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
                    
            /* calling the AES integrated function */
            /*-------------------------------------*/ 
            if (TST_OutputLocation == 0 )/* output buffer != input buffer    */
            {   
                TST_Status = CRYS_AES(TST_AES_MAC_Acceptance_TestData[TST_index].TST_IvCounter,
                                         TST_AES_MAC_Acceptance_TestData[TST_index].TST_Key,
                                         TST_AES_MAC_Acceptance_TestData[TST_index].TST_KeySize,
                                         TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                         TST_AES_MAC_Acceptance_TestData[TST_index].TST_OperationMode,
                                         TST_AES_MAC_Acceptance_TestData[TST_index].TST_input_data, 
                                         TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize, 
                                         TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data);                                
            }
            else /* input buffer = output buffer */ 
            {
                for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                {   
                    /*Copying the input buffer onto output buffer not to overwrite test data*/
                    DX_VOS_FastMemCpy(TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                   TST_AES_MAC_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                   TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                }       
                
                    TST_Status = CRYS_AES(TST_AES_MAC_Acceptance_TestData[TST_index].TST_IvCounter,
                                             TST_AES_MAC_Acceptance_TestData[TST_index].TST_Key,
                                             TST_AES_MAC_Acceptance_TestData[TST_index].TST_KeySize,
                                             TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                             TST_AES_MAC_Acceptance_TestData[TST_index].TST_OperationMode,
                                             TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data,
                                             TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize,
                                             TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data);           
                } 
     
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
    
            
            
            /* Debug massages */
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"Input Data Size [byte] = %d \n" ,TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize));
                        
            /* Compare the result to expected result */ 
            /*---------------------------------------*/ 
            for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
            {
                if (TST_AES_MAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC[Index] !=
                    TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data[Index])
                        TST_Status = TESTFAIL;
            }
    
            /*Print the outputted and then the expected result after mem compare */
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_AES_MAC_Acceptance_TestData[TST_index].TST_output_data, 
                                        TST_AES_MAC_Acceptance_TestData[TST_index].TST_dataSize); 
                                        
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\n Data Expected by engine [expected buffer]  -- \n",
                                        TST_AES_MAC_Acceptance_TestData[TST_index].TST_Expected_CBC_MAC,
                                        TST_CBC_MAC_SIZE);                  


            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_Memcompare",TST_AES_MAC_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
    
            
        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;

  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Acceptance_KeyWrapUnwarp      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES wrap unwrap  API's
 *  This function wraps and unwraps known vectors using AES
 *  Vector sizes used (in bytes) - 16/32/512                                 
 *  Key Size used (in bits)     - 128/192/256                               
 * Algorithm:       
 * 1. Call the CRYS_Wrap and Unwrap function to use the AES wrap/unwrap process
 * 2. Compare the outputted result to the expected one                                                      
 *
 ******************************************************************************/    
CRYSError_t CRYS_ATP_TST_AES_Acceptance_KeyWrapUnwarp ( void ) 
{
    /********* TST Variables Definitions ************/
    int              TST_index; 
    DxUint32_t       TST_WrapdDataActualSize,TST_OutputDataActualSize;
    CRYSError_t      TST_Status=0;
    char             TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_KeyWrapUnwarp";
    DxBool_t      Is_SecretKey=DX_FALSE;
  /********* TST Variables initialization *********/
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n")); 
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ** CRYS_ATP_TST_AES_Acceptance_KeyWrap Tests     ********************* \n"));     
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n \n"));
    
    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_index =0 ; TST_index < TST_NumberOfKeyWrapAcceptanceTests ; TST_index++) /*To run on all test vectors*/
    { 
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ************************************************************************ \n"));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    %s [test 5.%d]   **\n",TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Name,TST_index));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ************************************************************************ \n"));

        TST_Status = TESTPASS;

        /* No to effect data in the struct*/
        TST_WrapdDataActualSize = TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdDataSize;

        /* calling the AES Wrap function */
        /*-------------------------------*/ 
    if (TST_USE_SECRETKEY_TESTS) 
              Is_SecretKey=DX_TRUE;

    TST_Status = CRYS_AES_Wrap(TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_InputData, 
                                      TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_InputDataSize, 
                                      TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Key,
                                      TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_KeySize,
                                      (DxInt8_t)Is_SecretKey, 
                                      TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdData,
                                      &TST_WrapdDataActualSize);

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS AES Wrap",TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
   
  
        
        /* Compare the result to expected result */ 
        /*---------------------------------------*/                                                             
        TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdData ,
                                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_ExpectedWrapedData ,
                                                   TST_WrapdDataActualSize);
   
        /*Print the outputted and then the expected result after mem compare */
        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                   "\nData outputted by engine - TST_WrapdData -- \n",
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdData, 
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdDataSize); 

        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                   "\nData expected - ExpectedWrapedData -- \n",
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_ExpectedWrapedData, 
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdDataSize); 

        
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
  

        /* No to effect data in the struct*/
        TST_OutputDataActualSize = TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_InputDataSize;

        /* calling the AES Unwrap function */
        /*---------------------------------*/ 
        TST_Status = CRYS_AES_Unwrap(TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdData,
                                         TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_WrapdDataSize, 
                                         TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Key,
                                         TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_KeySize,
                     0, /* temporary! change it to isSecretKey variable */
                                         TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_OutputData,
                                         &TST_OutputDataActualSize);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS AES Unwrap",TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
   


        /* Compare the result to expected result */ 
        /*---------------------------------------*/                                                             
        TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_OutputData ,
                                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_InputData ,
                                                   TST_OutputDataActualSize);
   
        /*Print the outputted and then the expected result after mem compare */
        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                   "\nData outputted by engine - TST_OutputData -- \n",
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_OutputData, 
                                   TST_OutputDataActualSize); 

        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                   "\nData expected - TST_InputData -- \n",
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_InputData, 
                                   TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_InputDataSize); 

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_ARS_KeyWrap_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

  
    }/*End of TST_index*/     
  return TST_Status;

  EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/* End of CRYS_ATP_TST_AES_Acceptance_KeyWrapUnwarp */
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
#if TST_USE_SECRETKEY_TESTS
/*****************************************************************************
* Function Name:
*  CRYS_ATP_TST_AES_Acceptance_SK
*
* Inputs:
*  None
*
* Outputs:
*  CRYSError_t - Function error return
*
* Description:
*  This function Is an acceptance test for AES using the Secret key API's
*  This function encrypts and then decrypts known vectors using AES  
*  and then verifies that the result is the same as the input.
*  AES modes                    - CBC/ECB
*  Vector sizes used (in bytes) - 16/32/512/1536
*  Key used                     - Internal 128 bit
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:
* 1. Call the CRYS_AES_SecretInit function to start the encryption process
* 2. Call the CRYS_AES_Block - This function is called N times
*                              (N = number of blocks in data each block is 16byte)
* 3. Call the CRYS_AES_Free to end the encryption process
* 4. Call The CRYS_AES_SecretKey to decrypt the output of the encryption process.
*    The decrypt operation in done in one block.
* 5. Compare the outputted result to the initial inputed one one
*
******************************************************************************/
CRYSError_t CRYS_ATP_TST_AES_Acceptance_SK( void )
{

    /********* TST Variables Definitions ************/
    int                     TST_index;
    int                     Block_index ,NumOfBlocks;
    CRYSError_t             TST_Status=0;
    CRYS_AESUserContext_t   TST_ContextID_ptr;
    char                    TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Acceptance_SK";


    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_SK Tests ********************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************* \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_index =0 ; TST_index < TST_NumberOf_AES_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 6.%d] \n",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_index ));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 

       TST_Status = TESTPASS;

       /* Encrypt Operation */
       /*-------------------*/
       /*-------------------*/
       /* calling the AES init function */
       /*-------------------------------*/
       TST_Status = CRYS_AES_InitSecretKey(&TST_ContextID_ptr,
                                              TST_AES_Acceptance_TestData[TST_index].TST_IvCounter,
                                              CRYS_AES_Encrypt, 
                                              TST_AES_Acceptance_TestData[TST_index].TST_OperationMode);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_InitSecretKey - For encrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);


       /* calling the AES Block function */
       /*--------------------------------*/
       NumOfBlocks = TST_AES_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;

       /* Debug massages */
       ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %d \n",NumOfBlocks));
       ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %d \n" ,TST_AES_Acceptance_TestData[TST_index].TST_dataSize));
       ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %d \n",TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS));

       for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*AES block loop*/
       {
          TST_Status = CRYS_AES_Block(&TST_ContextID_ptr,
                                         TST_AES_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                         TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t),
                                         TST_AES_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block - Encrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);




       }/*End AES block loop*/

       /* calling the Free function */
       /*---------------------------*/
       TST_Status = CRYS_AES_Free( &TST_ContextID_ptr );

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Free - Encrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);


       /* Decrypt Operation */
       /*-------------------*/
       /*-------------------*/

       /* calling the AES SK  function */
       /*------------------------------*/
       if (TST_AES_Acceptance_TestData[TST_index].TST_OperationMode != CRYS_AES_CTR_mode)
       {       
            TST_Status = CRYS_AES_SecretKey(TST_AES_Acceptance_TestData[TST_index].TST_IvCounter,
                                                TST_EncryptMode[CRYS_AES_Decrypt],
                                                TST_AES_Acceptance_TestData[TST_index].TST_OperationMode,
                                                TST_AES_Acceptance_TestData[TST_index].TST_output_data,
                                                TST_AES_Acceptance_TestData[TST_index].TST_dataSize,
                                                TST_AES_Acceptance_TestData[TST_index].TST_output_data);
       }
       else
       {     
              TST_Status = CRYS_AES_SecretKey(TST_AES_Acceptance_TestData[TST_index].TST_IvCounter,
                                                 TST_EncryptMode[CRYS_AES_Encrypt],
                                                 TST_AES_Acceptance_TestData[TST_index].TST_OperationMode,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_output_data,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_dataSize,
                                                 TST_AES_Acceptance_TestData[TST_index].TST_output_data);
       }

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_SecretKey - For Decrypt",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);




       /* Compare the result to expected result */
       /*---------------------------------------*/
       TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_TestData[TST_index].TST_input_data ,
                                                  TST_AES_Acceptance_TestData[TST_index].TST_output_data,
                                                  TST_AES_Acceptance_TestData[TST_index].TST_dataSize);


       /*Print the outputted and then the expected result after mem compare */
       ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                   "\nData outputted by engine [output buffer - Decrypted]-- \n",
                                   TST_AES_Acceptance_TestData[TST_index].TST_output_data,
                                   TST_AES_Acceptance_TestData[TST_index].TST_dataSize);

       ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                  "\nData Expected by engine [input buffer]--\n",
                                  TST_AES_Acceptance_TestData[TST_index].TST_input_data,
                                  TST_AES_Acceptance_TestData[TST_index].TST_dataSize);

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/
  return TST_Status;

    EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/* End of CRYS_ATP_TST_AES_Acceptance_SK*/
#endif /*TST_USE_SECRETKEY_TESTS*/


#if TST_USE_PREDEFINED_SECRETKEY_TESTS
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_AES_Predefined_SecretKey       
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for AES integrated API                           
 *  This function Encrypts and Decrypts known vectors using AES secret key
 *  This test will pass only if the Secret key was programed to a pre-defined value!
 *
 *  AES modes                   - CBC/ECB
 *  Vector sizes used (in bytes) - 16/32/512                                 
 *  Key Size used (in bits)     - Secret Key                                
 *  This also outputs the results to an output buffer or the same input buffer
 *      
 * Algorithm:    
 * 1. Call the CRYS_AES function to use the AES encryption/decryption process
 * 2. Compare the outputted result to the expected one                                              
 *
 ******************************************************************************/    
CRYSError_t CRYS_ATP_TST_AES_Predefined_SecretKey ( void )
{
    /********* TST Variables Definitions ************/
    int              TST_index, NumOfBlocks,Block_index;
    int              TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
    char             TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_Predefined_SecretKey";
    CRYSError_t      TST_Status=0;

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ** CRYS_ATP_TST_AES_Predefined_SecretKey Tests using Predefined SCRETE KEY!!  ***** \n"));     
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n \n"));    
  
    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_AES_AcceptanceTestsSK ; TST_index++) /*To run on all test vectors*/
        {
            for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation 
                                                                                                                    TST_OperationMode = 0 == Encrypt
                                                                                                                     TST_OperationMode = 1 == Decrypt*/
            {           
                if (
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_ECB_mode
                        ||
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_CBC_mode
                        ||
                        (
                        TST_AES_Acceptance_TestData[TST_index].TST_OperationMode == CRYS_AES_CTR_mode &&
                        TST_ATP_EncryptDecryptFlag == TST_ENCRYPT /*CTR mode uses encrypt only!*/
                        )
                   )
                {       
                    if (TST_OutputLocation == 0)
                    {   
                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************* \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 7A.%d]  **\n",TST_AES_Acceptance_SK_TestData[TST_index].TST_Name,TST_index)); 
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   output buffer != input buffer                                      \n"));                  
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************* \n"));
                    }
                    else
                    {
                        /* output buffer == input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************************************************************ \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 7B.%d]  **\n",TST_AES_Acceptance_SK_TestData[TST_index].TST_Name,TST_index));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   output buffer == input buffer (output buffer)                         \n"));                       
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************************************************************ \n "));
                    }
                    
                    TST_Status = TESTPASS;
                    NumOfBlocks = TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
                            
                    /* calling the AES integrated function */
                    /*-------------------------------------*/
                    if (TST_OutputLocation == 0)/* output buffer != input buffer */
                    {
                        if (TST_ATP_EncryptDecryptFlag == 0) 
                        {
                            TST_Status = CRYS_AES_SecretKey(TST_AES_Acceptance_SK_TestData[TST_index].TST_IvCounter,
                                                               TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_OperationMode,
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_input_data,
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize,
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data);                               
                        }
                        else
                        {
                            TST_Status = CRYS_AES_SecretKey(TST_AES_Acceptance_SK_TestData[TST_index].TST_IvCounter,
                                                               TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_OperationMode,
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_Expected_output_data,
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize,
                                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data);                  
                        }                                    
                    }                   
                    else /* input buffer = output buffer */
                    {
                        for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
                        {      
                            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                            {
                            /*Copying the input buffer onto output buffer not to overwrite test data*/
                                DX_VOS_FastMemCpy(TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                               TST_AES_Acceptance_SK_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                               TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                            }
                            else
                            {
                            /*Copying the input buffer onto output buffer not to overwrite test data*/
                                DX_VOS_FastMemCpy(TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                           TST_AES_Acceptance_SK_TestData[TST_index].TST_Expected_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                                           TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
                            }                                      
                        }       
                        TST_Status = CRYS_AES_SecretKey(TST_AES_Acceptance_SK_TestData[TST_index].TST_IvCounter,
                                                       TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                                                       TST_AES_Acceptance_SK_TestData[TST_index].TST_OperationMode,
                                                       TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data,
                                                       TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize,
                                                       TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data);          
                    } 

                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
          {
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES Encrypt",TST_AES_Acceptance_SK_TestData[TST_index].TST_Name,TST_MasterTestName);
          }
                    else
          {
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES Decrypt",TST_AES_Acceptance_SK_TestData[TST_index].TST_Name,TST_MasterTestName);
          }
                        
                     
                            
                    /* Debug massages */
                    PLAT_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %1d \n" ,TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize));
                    /* Compare the result to expected result */
                    /*---------------------------------------*/                                                             
                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)  
                    {
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data ,
                                                           TST_AES_Acceptance_SK_TestData[TST_index].TST_Expected_output_data ,
                                                           TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize);
                    }
                    else
                    {
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data ,
                                                           TST_AES_Acceptance_SK_TestData[TST_index].TST_input_data,
                                                           TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize);             
                    }
        
                    /*Print the outputted and then the expected result after mem compare */
                    PLAT_LOG_TST_PRINT_DisplayBuffer((CRYS_ATP_TST_DBG_HIGH,
                                                      "\nData outputed by engine [output buffer]-- \n",
                                                      TST_AES_Acceptance_SK_TestData[TST_index].TST_output_data,
                                                      TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize));

                    if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
                    {
                        PLAT_LOG_TST_PRINT_DisplayBuffer((CRYS_ATP_TST_DBG_HIGH,
                                                          "\nData Expected by engine [expected buffer - encrypt]--\n",
                                                          TST_AES_Acceptance_SK_TestData[TST_index].TST_Expected_output_data,
                                                          TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize));                  
                    }
                    else
                    {
                        PLAT_LOG_TST_PRINT_DisplayBuffer((CRYS_ATP_TST_DBG_HIGH,
                                                          "\nData Expected by engine [input buffer - for decrypt]--\n",
                                                          TST_AES_Acceptance_SK_TestData[TST_index].TST_input_data,
                                                          TST_AES_Acceptance_SK_TestData[TST_index].TST_dataSize));                      
                    }               

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_Acceptance_SK_TestData[TST_index].TST_Name,TST_MasterTestName);
                   
                }/* End of first if statment*/
            }/* End of TST_EncryptDecryptFlag*/     
        }/*End of TST_index//TST_NumberOf_AES_AcceptanceTests*/     
    } /* End of TST_OutputLocation loop*/
  return TST_Status;
    EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/* end of CRYS_SYS_TST_AES_Acceptance_SK_Integrated*/
#endif /*TST_USE_PREDEFINED_SECRETKEY_TESTS*/

#ifndef CRYS_NO_AES_XTS_MODE_SUPPORT
/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_AES_XTS_Acceptance_NON_integrated      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES XTS non integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES
*  AES modes                   - CRYS_AES_XTS_mode   
*  Tweak sizes used (in bytes) - 16                                 
*  Key Size used (in bits)     - 256/512                               
*  
* Algorithm:       
* 1. Call the CRYS_AES_XTS_Init function to start the encryption/decryption process
* 2. Call the CRYS_AES_Block - This function is called N times 
*                              (N = number of blocks in data each block is 16byte) 
* 3. Call the CRYS_AES_XTS_Finish to end the encryption/decryption process 
* 4. Compare the outputted result to the expected one                                                   
*
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_AES_XTS_Acceptance_NON_integrated ( void ) 
{

  /********* TST Variables Definitions ************/
  int                     TST_index;
  int                     Block_index ,NumOfBlocks;
  CRYSError_t             TST_Status=0;
  int                     TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
  CRYS_AESUserContext_t   TST_ContextID_ptr;  
  char                    TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_XTS_Acceptance_NON_integrated"; 


  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"***************************************************************************** \n"));
  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_XTS_Acceptance_NON_integrated Tests ********************* \n"));      
  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"***************************************************************************** \n \n"));

  /************************************************/
  /*********** Start of test **********************/
  /************************************************/
  for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
  {
    for (TST_index =0 ; TST_index < TST_NumberOf_AES_XTS_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
      for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
                                                          TST_OperationMode = 0 == Encrypt
                                                          TST_OperationMode = 1 == Decrypt*/
      {                                                              
        if (TST_OutputLocation == 0) /*Select output buffer*/
        {   
          /* output buffer != input buffer    */
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 1A.%d] \n",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_index ));
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   output buffer != input buffer                                       \n"));         
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
        }
        else
        {
          /* output buffer == input buffer    */
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n")); 
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 1B.%d] \n",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_index));
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   output buffer == input buffer (output buffer)                       \n"));                             
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n "));
        }

        TST_Status = TESTPASS;

        /* calling the AES init function */ 
        /*-------------------------------*/ 
        TST_Status = CRYS_AES_XTS_Init(&TST_ContextID_ptr,
                          TST_AES_XTS_Acceptance_TestData[TST_index].TST_Tweak, 
                          TST_AES_XTS_Acceptance_TestData[TST_index].TST_Key,
                          TST_AES_XTS_Acceptance_TestData[TST_index].TST_KeySize,
                          TST_EncryptMode[TST_ATP_EncryptDecryptFlag]);

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS_Init",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
        
  

        /* calling the AES Block function */
        /*--------------------------------*/
        NumOfBlocks = TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;

        /* Debug massages */
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %d \n",NumOfBlocks));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %d \n" ,TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %d \n",TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS));


        if (TST_OutputLocation == 0)/* input buffer != output buffer */
        {
          for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*AES block loop*/ 
          {   
            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
            { 
              TST_Status = CRYS_AES_Block( &TST_ContextID_ptr,     
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                              TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); 

              TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block - Encrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);                             
            }
            else
            {                           
              TST_Status = CRYS_AES_Block( &TST_ContextID_ptr,     
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),         
                              TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); 

              TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block - Decrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
            }   
 

          }/*End AES block loop*/ 
        }
        else /* input buffer = output buffer */ 
        {
          for (Block_index =0 ; Block_index < NumOfBlocks  ; Block_index++)
          {
            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
            {
              /*Copying the input buffer onto output buffer not to overwrite test data*/
              DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                        TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                        TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
            }
            else
            {
              /*Copying the input buffer onto output buffer not to overwrite test data*/
              DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
                        TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index), 
                        TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
            }
          }           

          for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
          {   
            TST_Status = CRYS_AES_Block(&TST_ContextID_ptr,           
                            TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
                            TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
                            TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); 

            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
            {
              TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block Encrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
            }
            else
            {
              TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_Block Decrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
            }
            
                         
          }
        }    


        /* calling the Finish function */ 
        /*-----------------------------*/             

        /* Debug massages */        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Finish Input Data Size [byte] = %d \n" ,TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize));
        

        if (TST_OutputLocation == 0)/* input buffer != output buffer */
        {
          if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
          { 
            TST_Status = CRYS_AES_XTS_Finish(&TST_ContextID_ptr,     
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputData,        
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize, 
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS_Finish - Encrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);                             
          }
          else
          {                           
            TST_Status = CRYS_AES_XTS_Finish(&TST_ContextID_ptr,     
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishExpectedOutputData,         
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize, 
                              TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS_Finish - Decrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
          }   

            
        }
        else /* input buffer = output buffer */ 
        {          
          if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
          {
            /*Copying the input buffer onto output buffer not to overwrite test data*/
            DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData,
                      TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputData,
                      TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);                        
          }
          else
          {
            /*Copying the input buffer onto output buffer not to overwrite test data*/
            DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData,
                      TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishExpectedOutputData, 
                      TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);                          
          }
                

          
          TST_Status = CRYS_AES_XTS_Finish(&TST_ContextID_ptr,           
                            TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData,        
                            TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize,
                            TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData);        

          if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
          {
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS_Finish Encrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
          }
          else
          {
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS_Finish Decrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
          }

                  
        }    
          

        /* Compare the Block result to expected result */ 
        /*---------------------------------------------*/ 
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)  
        {          
          TST_Status = (CRYSError_t)DX_VOS_MemCmp( TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data ,
                                TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data, 
                                TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize);
        }
        else
        {
          TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data ,
                                 TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data,
                                 TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize);                
        }

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);                   


        /* Compare the Finish result to expected result */ 
        /*----------------------------------------------*/ 
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)  
        {
          TST_Status = (CRYSError_t)DX_VOS_MemCmp( TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData ,
                                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishExpectedOutputData, 
                                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);
        }
        else
        {
          TST_Status = (CRYSError_t)DX_VOS_MemCmp( TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData ,
                                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputData,
                                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);                
        }

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);                   

        
      }/* End of TST_ATP_EncryptDecryptFlag*/ 
    }/*End of TST_index//TST_NumberOf_AES_XTS_AcceptanceTests*/     
  } /* End of TST_OutputLocation loop*/
  return TST_Status;

EXIT_ON_ERROR:

  return DX_FAILURE;

}/* End of CRYS_ATP_TST_AES_XTS_Acceptance_NON_integrated*/ 

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_AES_XTS_Acceptance_Integrated      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES XTS non integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES
*  AES modes                   - CRYS_AES_XTS_mode   
*  Tweak sizes used (in bytes) - 16                                 
*  Key Size used (in bits)     - 256/512                               
*  
* Algorithm:       
* 1. Call the CRYS_AES_XTS function to encrypt/decrypt data
* 2. Compare the outputted result to the expected one                                                                                                       
*
******************************************************************************/    
CRYSError_t CRYS_ATP_TST_AES_XTS_Acceptance_Integrated ( void ) 
{

  /********* TST Variables Definitions ************/
  int              TST_index;   
  CRYSError_t      TST_Status=0;
  int              TST_OutputLocation; /* 0 >> input!=output buffer // 1 >> input=output buffer*/
  char             TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_XTS_Acceptance_Integrated";
  unsigned int         TempDataSize = 0;
  /********* TST Variables initialization *********/

  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n")); 
  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ** CRYS_ATP_TST_AES_XTS_Acceptance_Integrated Tests     ********************* \n"));     
  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************* \n \n"));

  /************************************************/
  /*********** Start of test **********************/
  /************************************************/
  for (TST_OutputLocation =0 ; TST_OutputLocation < 2 ;TST_OutputLocation++) 
  {
    for (TST_index =0 ; TST_index < TST_NumberOf_AES_XTS_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
      /* Concatenate Finish data to Block data and put the result in block data */      
      DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data+TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize,
                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputData, 
                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);

      DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data+TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize,
                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishOutputData, 
                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);

      DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data+TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize,
                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishExpectedOutputData, 
                TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize);
      
      /* Update data size to TempDataSize variable*/      
      TempDataSize = TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize+TST_AES_XTS_Acceptance_TestData[TST_index].TST_FinishInputDataSize;

      for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++) /*Select mode or operation  
                                                        TST_OperationMode = 0 == Encrypt
                                                        TST_OperationMode = 1 == Decrypt*/
      {                

        if (TST_OutputLocation == 0)/*Select output buffer*/ 
        {   
          /* output buffer != input buffer    */
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************** \n"));
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    %s [test 2A.%d]  **\n",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_index));   
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    output buffer != input buffer                                       \n"));                    
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ********************************************************************** \n"));
        }
        else
        {
          /* output buffer == input buffer    */
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ************************************************************************ \n"));
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    %s [test 2B.%d]   **\n",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_index));
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"    output buffer == input buffer (output buffer)                         \n"));                      
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," ************************************************************************ \n"));
        }

        TST_Status = TESTPASS;        

        /* calling the AES integrated function */
        /*-------------------------------------*/ 
        if (TST_OutputLocation == 0)/* input buffer != output buffer */
        {
          if (TST_ATP_EncryptDecryptFlag ==0) 
          {
            TST_ATP_StartPerformanceTimer(&TST_Timer);

            TST_Status = CRYS_AES_XTS(TST_AES_XTS_Acceptance_TestData[TST_index].TST_Tweak,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_Key,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_KeySize,
                           TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data,               
                           TempDataSize,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data); 

            TST_ATP_GetPerformanceTimer(&TST_Timer);
            TST_ATP_CalcPerformance(TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize);
          }
          else
          {
            TST_ATP_StartPerformanceTimer(&TST_Timer);

            TST_Status = CRYS_AES_XTS(TST_AES_XTS_Acceptance_TestData[TST_index].TST_Tweak,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_Key,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_KeySize,
                           TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data,              
                           TempDataSize,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data);                   

              TST_ATP_GetPerformanceTimer(&TST_Timer); 
              TST_ATP_CalcPerformance(TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize);
            }
          }
          else /* input buffer = output buffer */ 
          {            
            if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
            {
              /*Copying the input buffer onto output buffer not to overwrite test data*/
              DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data,
                        TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data,                
                        TempDataSize);  
            }
            else
            {
              /*Copying the input buffer onto output buffer not to overwrite test data*/
              DX_VOS_FastMemCpy(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data,
                        TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data,                 
                        TempDataSize);  
            }
          
            TST_ATP_StartPerformanceTimer(&TST_Timer);

            TST_Status = CRYS_AES_XTS(TST_AES_XTS_Acceptance_TestData[TST_index].TST_Tweak,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_Key,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_KeySize,
                           TST_EncryptMode[TST_ATP_EncryptDecryptFlag],
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data,              
                           TempDataSize,
                           TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data);

            TST_ATP_GetPerformanceTimer(&TST_Timer);
            TST_ATP_CalcPerformance(TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize);
          } 

          if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
          {
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS Encrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
          }
          else
          {
            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_XTS Decrypt",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName); 
          }

      

          /* Debug massages */
          ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"Input Data Size [byte] = %d \n" ,TST_AES_XTS_Acceptance_TestData[TST_index].TST_dataSize));

          /* Compare the result to expected result */ 
          /*---------------------------------------*/                                                             
          if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)  
          {
            TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data ,
                                   TST_AES_XTS_Acceptance_TestData[TST_index].TST_Expected_output_data ,              
                                   TempDataSize);
          }
          else
          {
            TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_AES_XTS_Acceptance_TestData[TST_index].TST_output_data ,
                                   TST_AES_XTS_Acceptance_TestData[TST_index].TST_input_data,              
                                   TempDataSize);                
          }

          
          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp",TST_AES_XTS_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

           
        
      }/* End of TST_ATP_EncryptDecryptFlag*/ 
    }/*End of TST_index//TST_NumberOf_AES_XTS_AcceptanceTests*/     
  } /* End of TST_OutputLocation loop*/
  return TST_Status;

EXIT_ON_ERROR:

  return DX_FAILURE;


}
#endif
#ifndef CRYS_NO_AESCCM_SUPPORT
/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_AES_CCM_Acceptance_Non_Integrated      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is an Acceptance test for AES CCM non integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES_CCM
*  AES modes                   - CRYS_AES_CCM_mode   
*  Key sizes used (in bytes) - 16                                 
*  Adata Size used (in bytes)- 16/500
*  Text data size( in bytes)- 16/500/1024
*  Nonce size ( in bytes) - 7 - 13 
*  T mac ( in bytes) - 4/6/8/10/12/14
*  Q fields( in bytes)- 2/3/4/8

* Algorithm:       
* 1. Call the CRYS_AES_CCM_init to init the context
* 2. Call the CRYS_AESCCM_BlockAdata function.
* 3. Copy the input buffer to the output buffer
* 4. Call the CRYS_AESCCM_BlockText data function, without the last block
* 5. Call the CRYS_AESCCM_BlockLastTextData function.
* 6. Call the CRYS_AESCCM_Finish function
* 7. Compare the output data to the expected data
******************************************************************************/    

CRYSError_t CRYS_ATP_TST_AES_CCM_Acceptance_NON_integrated ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index = 0;
    int                         sizeOfT = 0;
    int                         Block_index = 0 ,NumOfBlocks = 0 , BlockSize = 0;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_CCM_Acceptance_NON_integrated"; 
    CRYSError_t                 TST_Status=0;
    CRYS_AESCCM_UserContext_t   TST_ContextID;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*****************************************************************************\n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_NON_integrated CCM Tests *********************\n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*****************************************************************************\n\n")); 

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
  
    /*To run on all test vectors*/
    for (TST_index =0 ; TST_index < TST_NumberOfAESCCMAcceptanceTests ; TST_index++) 
    { /*Select mode or operation */
      for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)
      {    

        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3A.%d]\n",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name  ,TST_index )); 
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer != input buffer                                     \n"));        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));

        /* calling the AES_CCM init function */ 
        /*----------------------------------*/ 
        if(TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
        {
           TST_Status = CRYS_AESCCM_Init( 
					                                  &TST_ContextID,
					                                   TST_ATP_EncryptDecryptFlag,                          /*TST_OperationMode = 1 == Decrypt*/*
                                            &TST_AES_CCM_Acceptance_TestData[TST_index].CCM_Key,   /*AESCCM key*/            
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize, /*full size of additional data*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize, /*full size of text data*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].QFieldSize,/*size of Q field in control block*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].N_ptr,     /*Nonce*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfN,   /*size of N buffer*/
				                                     TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT );   /*size of CCM-MAC (T) */  
        }
        /*Decrypt*/
        else
        {
           TST_Status = CRYS_AESCCM_Init( 
					                                  &TST_ContextID,
					                                   TST_ATP_EncryptDecryptFlag,                          /*TST_OperationMode = 1 == Decrypt*/*
                                            &TST_AES_CCM_Acceptance_TestData[TST_index].CCM_Key,   /*AESCCM key*/            
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize, /*full size of additional data*/
                                             TST_AES_CCM_Acceptance_TestData[TST_index].OutputDataSize, /*full size of text data*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].QFieldSize,/*size of Q field in control block*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].N_ptr,     /*Nonce*/
					                                   TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfN,   /*size of N buffer*/
				                                     TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT );   /*size of CCM-MAC (T) */  
        }

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"AES_CCM_Init",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);


        /* calling the CRYS_AESCCM_BlockAdata function */
        /*--------------------------------------------*/
        NumOfBlocks = TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS; 
        BlockSize = TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
        /*case that the buffer is not multiple of 16*/
        if(TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS != 0)
        {
           NumOfBlocks++;
        }

        /* Debug massages */
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %d \n",NumOfBlocks));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %d \n" ,TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %d \n",TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS));

        /*Running in loop to block the Adata buffer in parts of TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS */        
        for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++) 
        {   
           if((Block_index == (NumOfBlocks - 1)) && (Block_index != 0) )//last block size
           {
              BlockSize =  (TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);
           }
         /*Block Adata*/
         TST_Status = CRYS_AESCCM_BlockAdata( 
                                               &TST_ContextID,   
                                                TST_AES_CCM_Acceptance_TestData[TST_index].AdataData + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS * Block_index),
                                                BlockSize );     

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AESCCM_BlockAdata",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
        }

        /* calling the CRYS_AESCCM_BlockTextData function */
        /*------------------------------------------------*/
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
        {
          /*Copying the input buffer onto output buffer not to overwrite test data*/
          DX_VOS_FastMemCpy(TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,
                            TST_AES_CCM_Acceptance_TestData[TST_index].TextData,
                            TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize);
        }

        NumOfBlocks = (TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize / TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS); 
        /*case that the buffer is not multiple of 16*/
        if((TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS) != 0)
         {
           NumOfBlocks++;
         }
        BlockSize   = TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;
        
        /*AES block loop without the last block*/
        for (Block_index = 0 ; Block_index < (NumOfBlocks - 1) ; Block_index++) 
        {  
          TST_Status =  CRYS_AESCCM_BlockTextData(
                                                    &TST_ContextID,   
                                                    TST_AES_CCM_Acceptance_TestData[TST_index].OutputData + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS * Block_index),
            	                                      BlockSize,
            	                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS * Block_index) );
          

          if (TST_ATP_EncryptDecryptFlag == TST_DECRYPT)
          {
            if((TST_ATP_CheckErrorReturn(TST_Status,"CRYS_AESCCM_BlockTextData - DECRYPT",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName) == TST_FAIL))
              {
                return DX_ERROR;
              }
          } 
          else
          {
            if((TST_ATP_CheckErrorReturn(TST_Status,"CRYS_AESCCM_BlockTextData - ENCRYPT",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName) == TST_FAIL))
              {
                return DX_ERROR;
              }
          }
        }

        /* calling the CRYS_AESCCM_BlockLastTextData function */
        /*---------------------------------------------------*/

        /*calculate the size of the last block*/
        if(TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
        {
           if(TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS != 0)
           {
             BlockSize =  (TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);
           }
        }
        /*Decrypt*/
        else
        {
           if(TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS != 0)
           {
             BlockSize =  (TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize % TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);
           } 
           BlockSize = BlockSize + TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT;
        }
              
        /*Calculate size of Tmac
          in case of encrypt need to add this number
          in case of decrypt need to subtract this number*/    
        if(TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)//DELETE
        {
          sizeOfT = (TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT);
        }
        /*Decrypt*/
        else
        {
          sizeOfT = -(TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT);
        }

        /*Block the last text data */
        TST_Status = CRYS_AESCCM_BlockLastTextData(
							                        &TST_ContextID,   
                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS * Block_index),
							                        BlockSize,
							                        TST_AES_CCM_Acceptance_TestData[TST_index].OutputData + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS * Block_index),
							                        BlockSize + sizeOfT);
          
        if (TST_ATP_EncryptDecryptFlag == TST_DECRYPT)
        {
          if((TST_ATP_CheckErrorReturn(TST_Status,"CRYS_AESCCM_BlockLastTextData - DECRYPT",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName) == TST_FAIL))
          {
            return DX_ERROR;
          }
        }  

        else
        {
          if((TST_ATP_CheckErrorReturn(TST_Status,"CRYS_AESCCM_BlockLastTextData - ENCRYPT",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName) == TST_FAIL))
          {
            return DX_ERROR;
          }
        }
        
        /* calling the Finish function */ 
        /*----------------------------*/     
        TST_Status = CRYS_AESCCM_Finish(&TST_ContextID);  
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES_CCM_Free",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

        /* Compare the result to expected result */ 
        /*---------------------------------------*/ 
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
        {
           TST_Status = (CRYSError_t)DX_VOS_MemCmp(
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].ExpectedOutputData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputDataSize); 
        }
        /*Decrypt*/
        else
        {
           TST_Status = (CRYSError_t)DX_VOS_MemCmp(
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].TextData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize);                
        }

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_MemCompare",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

      }/* End of TST_ATP_EncryptDecryptFlag*/ 
    }/*End of TST_index TST_NumberOf_AES_CCM_AcceptanceTests*/ 
EXIT_ON_ERROR:

return TST_Status;

}/* End of CRYS_ATP_TST_AES_CCM_Acceptance_Integrated */


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_AES_CCM_Acceptance_Integrated      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES CCM integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES_CCM
*  AES modes                   - CRYS_AES_CCM_mode   
*  Key sizes used (in bytes) - 16                                 
*  Adata Size used (in bytes)- 16/500
*  Text data size( in bytes)- 16/500/1024
*  Nonce size ( in bytes) - 7 - 13 
*  T mac ( in bytes) - 4/6/8/10/12/14
*  Q fields( in bytes)- 2/3/4/8

* Algorithm:       
* 1. Call the CRYS_AES_CCM function to encrypt/decrypt data
* 2. Compare the outputted result to the expected one  
******************************************************************************/   
CRYSError_t CRYS_ATP_TST_AES_CCM_Acceptance_Integrated ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index = 0;
    int                         Block_index = 0;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_AES_CCM_Acceptance_NON_integrated"; 
    CRYSError_t                 TST_Status=0;
 
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*****************************************************************************\n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_AES_Acceptance_NON_integrated CCM Tests *********************\n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*****************************************************************************\n\n")); 

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
  
    /*To run on all test vectors*/
    for (TST_index =0 ; TST_index < TST_NumberOfAESCCMAcceptanceTests ; TST_index++) 
    { /*Select mode or operation */
      for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)
      {    

        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s[test 3A.%d]\n",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name  ,TST_index )); 
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  output buffer != input buffer                                     \n"));        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************************************************************\n"));
        /*Call the CRYS_AES_CCM function to encrypt*/
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
        {
          TST_Status = CRYS_AESCCM( 
					                            TST_ATP_EncryptDecryptFlag,   /* CCM encrypt-decrypt mode*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].CCM_Key,        /*AES-CCM key*/            
					                            TST_AES_CCM_Acceptance_TestData[TST_index].QFieldSize,     /*size of Q field in control block*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].N_ptr,          /*Nonce*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfN,        /*size of N buffer*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].AdataData,    /*input data pointer*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize,    /*input data size*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].TextData, /*input data pointer*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize, /*input data size*/
                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,/*output data pointer*/
                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputDataSize,/*output data size*/
                                      TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT  );      /* size of CCM-MAC */
        }
        /*Call the CRYS_AES_CCM function to decrypt*/
        else
        {
          TST_Status = CRYS_AESCCM( 
					                            TST_ATP_EncryptDecryptFlag,   /* CCM encrypt-decrypt mode*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].CCM_Key,        /*AES-CCM key*/            
					                            TST_AES_CCM_Acceptance_TestData[TST_index].QFieldSize,     /*size of Q field in control block*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].N_ptr,          /*Nonce*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfN,        /*size of N buffer*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].AdataData,    /*input data pointer*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].AdataSize,    /*input data size*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].OutputData, /*input data pointer*/
					                            TST_AES_CCM_Acceptance_TestData[TST_index].OutputDataSize, /*input data size*/
                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,/*output data pointer*/
                                      TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize,/*output data size*/
                                      TST_AES_CCM_Acceptance_TestData[TST_index].SizeOfT  );      /* size of CCM-MAC */
        }
                       
        /* Compare the result to expected result */ 
        /*---------------------------------------*/ 
        if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
        {
           TST_Status = (CRYSError_t)DX_VOS_MemCmp(
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].ExpectedOutputData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputDataSize); 
        }
        /*Decrypt*/
        else
        {
           TST_Status = (CRYSError_t)DX_VOS_MemCmp(
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].TextData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].OutputData,
                                                      TST_AES_CCM_Acceptance_TestData[TST_index].TextdataInputSize);                
        }

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_MemCompare",TST_AES_CCM_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
      }/* End of TST_ATP_EncryptDecryptFlag*/ 
    }/*End of TST_index TST_NumberOf_AES_CCM_AcceptanceTests*/ 
EXIT_ON_ERROR:
return TST_Status;


}/* End of CRYS_ATP_TST_AES_CCM_Acceptance_Integrated */
#endif /*CRYS_NO_AESCCM_SUPPORT*/ 
#endif

